## -*-Tcl-*- (install) (nowrap)
## 
 # This file : filtersMenu.tcl
 # Created : 2000-04-03 14:00:56
 # Last modification : 2001-11-03 00:42:05
 # Author : Bernard Desgraupes
 # e-mail : <berdesg@easynet.fr>
 # Web-page : <http://webperso.easyconnect.fr/berdesg/alpha.html>
 # Description :
 #      This is a menu/feature for Alpha. It allows you  to  do,  in  one
 #      run and very quickly, multiple and successive "find and  replace"
 #      operations which are stored in files called filters. There can be
 #      as many as you wish different "find and replace" in  any  filter.
 #      These filters apply to a selection, a file or an entire folder.
 # 
 #      You can very easily create your own filters.
 # 
 #      See the doc in the Filters Help (it is in the Help menu once  the
 #      package is installed).
 # 
 # (c) Copyright : Bernard Desgraupes 2000, 2001
 # This is free software. See licensing terms in the Filters Help file.
 ##

alpha::menu filtersMenu 1.5.3 global "301" {
    addMode Fltr filtersMenu {*.flt} {filtersMenu}
    package::addPrefsDialog filtersMenu
} {
    filtersMenu
    set names [list applyToSelection applyToFile applyMultiToSelection applyMultiToFile \
      applyTempToSelection applyTempToFile]
    foreach i $names {
	hook::register requireOpenWindowsHook [list "301" $i] 1
	hook::register requireOpenWindowsHook [list filtersUtilities checkSyntax] 1
    }
} {
    set names [list applyToSelection applyToFile applyMultiToSelection applyMultiToFile \
      applyTempToSelection applyTempToFile]
    foreach i $names {
	hook::deregister requireOpenWindowsHook [list "301" $i] 1
	hook::deregister requireOpenWindowsHook [list filtersUtilities checkSyntax] 1
    }
} uninstall {
    file delete -force [file join $HOME Tcl Menus "Filters Menu"]
    file delete [file join $HOME Help "Filters Help"]
} maintainer {
    "Bernard Desgraupes" <berdesg@easynet.fr> <http://webperso.easyconnect.fr/berdesg/> 
}  help {file "Filters Help"}

namespace eval flt {}

# # # Filters menu preferences # # #

# Max number of errors allowed when checking filter's syntax.
# Will stop checking if more.
newPref variable maxNumbErr 15 filtersMenu
# When applying filters to a selection, it is faster to use a scrap window.
# But for a short selection, you could prefer the filtering to be applied directly.
# Define here what is a "short" selection.
newPref variable maxBeforeScrap 70 filtersMenu
# To warn, before applying a filter to a whole file,
# that it is a non-undoable action.
newPref flag warnUndoable 1 filtersMenu
# To display a short description of the filters syntax in the Temporary Filter and
# in new filters.
newPref flag showFilterSyntax 1 filtersMenu
if {${alpha::platform} == "alpha"} {
# Filtering runs much faster if superSearch is not active. If this flag is checked,
# superSearch (if present) will be deactivated during filtering operations and then 
# immediatly reactivated. If unchecked, filtering will be slowed down.
newPref flag ToggleSupSearch 1 filtersMenu
} else {
    # Alphatk always uses supersearch.
    set filtersMenumodeVars(ToggleSupSearch) 0
}

# # # Preferences for the Fltr mode # # #
# Fltr mode is defined internally to allow specific features for the filters files, 
# such as coloring of comments, proper handling of the comment/uncomment procs,  
# option-click on title bar... 
# There is no reason to change the following prefs which are most standard.
newPref f electricTab {0} Fltr
newPref f wordWrap {0} Fltr
newPref v leftFillColumn {0} Fltr
newPref v fillColumn {75} Fltr
newPref v prefixString {!! } Fltr
newPref v wordBreak {[a-zA-Z0-9]+} Fltr
newPref v wordBreakPreface {[^a-zA-Z0-9]} Fltr
# Default color for the comments in the filters :
newPref v commentColor red Fltr

regModeKeywords -e {!!} -c $FltrmodeVars(commentColor) Fltr {}


# # # Initialisation of some variables # # #

# The current filter's name
set flt_params(currFiltername) ""
# Create a folder for the filters if it does not exist already.
# It is located in Alpha :Tcl:Menus:Filters Menu:
if {![file exists [file join $HOME Tcl Menus "Filters Menu" Filters]]} {
	file mkdir [file join $HOME Tcl Menus "Filters Menu" Filters]}
# Path to the filters folder
set flt_params(dirpath) "[file join $HOME Tcl Menus "Filters Menu" Filters]"
# List of the filters (full paths) in the filters folder
set flt_params(filtersindir) [glob -nocomplain -dir  $flt_params(dirpath) *.flt]
# Build the list of all the filters names
set flt_params(filtnameindir) ""
foreach filt $flt_params(filtersindir) {lappend flt_params(filtnameindir) "[file rootname [file tail $filt]]"}
# Initialise the list of filters contained in the multi filter
set flt_params(multiList) ""
# Name of the Temporary filter (to do filtering on the fly). It is written to the disk 
# but deleted and recreated each time Alpha is launched
set flt_params(tempfltname) [file::makeNameLegal "* Temporary Filter *"]
if {[file exists [file join $flt_params(dirpath) $flt_params(tempfltname)]]} {
    file delete [file join $flt_params(dirpath) $flt_params(tempfltname)]
}
# Name of the window containing the results of a syntax checking
set flt_params(errfltname) "* Syntax Checking Results *"
if {[file exists [file join $flt_params(dirpath) $flt_params(errfltname)]]} {
    file delete [file join $flt_params(dirpath) $flt_params(errfltname)]
}
# Name of the window displaying the syntax for filters
set flt_params(syntfltname) "* Filters Syntax *"
# Name of the scrap window used for fast filtering of a selection
set flt_params(scrapname) "* Filters Scrap Window *"
# Variable to avoid the "not undoable" warning to be displayed each time when using a multifilter
set flt_params(firstapplic) 0
# Used by the scanning procs :
set flt_case(0) "--"
set flt_case(1) "-nocase"
# Prefix to dimm/undimm menu items :
set flt_params(prfx) "("
if {${alpha::platform} == "alpha"} {
    # Variable to remember to turn superSearch back on if ever it was turned off :
    set flt_params(supseatog) 0
}

# Explanation of the filters' syntax to be inserted, as a reminder, at the beginning of every new filter
# and in the Temporary Filter. There is a flag to cancel this message.
set flt_params(usage) "searchString    replacementString    \[option\]"
set flt_params(filtersyntax) "!! SYNTAX : \n"
append flt_params(filtersyntax) "!!  $flt_params(usage)\n"
append flt_params(filtersyntax) "!!    where the three arguments are separated by one or more tabulations.\n"
append flt_params(filtersyntax) "!!    The option is a (possibly empty) string containing 0 or 1 and/or\n"
append flt_params(filtersyntax) "!!    one of the letters i and m with the following signification :\n"
append flt_params(filtersyntax) "!!        0 (or nothing) for an ordinary textual search (this is the default)\n"
append flt_params(filtersyntax) "!!        1 for a search with regular expressions\n"
append flt_params(filtersyntax) "!!        i for a case insensitive search\n"
append flt_params(filtersyntax) "!!        m to match words exactly (not only a substring of a word)\n"
append flt_params(filtersyntax) "!!    The options can be combined in any order : 0m, im1, i, 0m etc.\n"
append flt_params(filtersyntax) "!!    Put as many of these instructions as you want in your filter.\n"
append flt_params(filtersyntax) "!!    Each filtering instruction must be on a single line.\n"
append flt_params(filtersyntax) "!!    A line starting with two exclamation signs is considered a comment\n"
append flt_params(filtersyntax) "!!    and not a filter instruction.\n\n\n"

# For the Fltr mode :
set Fltr::commentCharacters(General) "!! "
set Fltr::commentCharacters(Paragraph) [list "!!!! " "!!!!" "!! "]
    
proc filtersMenu {} {}


# # # Menu declarations # # #

menu::buildProc filtersMenu menu::buildfiltersMenu
menu::buildProc filtersUtilities menu::buildFiltersUtilities


# # # Building procedures # # #

proc menu::buildfiltersMenu {} {
    global filtersMenu flt_params
    set flt_params(prfx) [expr {[llength [winNames]] ? "" : "(" }]
    set ma ""
    lappend ma "<E<SpickAFilter"
    lappend ma "<S<IbuildAMultiFilter"
    lappend ma "(-"
    lappend ma "<E<S$flt_params(prfx)applyToSelection"
    lappend ma "<S<I$flt_params(prfx)applyMultiToSelection"
    lappend ma "<E<S$flt_params(prfx)applyToFile"
    lappend ma "<S<I$flt_params(prfx)applyMultiToFile"
    lappend ma "<E<SapplyToFolder..."
    lappend ma "<S<IapplyMultiToFolder..."
    lappend ma "(-"
    lappend ma "temporaryFilter"
    lappend ma "$flt_params(prfx)applyTempToSelection"
    lappend ma "$flt_params(prfx)applyTempToFile"
    lappend ma "applyTempToFolder..."
    lappend ma "(-"
    lappend ma [list Menu -n filtersUtilities {}]
    lappend ma "(-"
    if {$flt_params(currFiltername) != ""} {
	set flt_params(currFiltername) [file tail $flt_params(currFiltername)]
	lappend ma  [menu::itemWithIcon "currentFilter" 83] 
	lappend ma " $flt_params(currFiltername)"
    } else {
	lappend ma [menu::itemWithIcon "noFilterSelected" 82] 
    }	
    
    return [list build $ma flt::MenuProc {filtersUtilities} $filtersMenu]
}

proc menu::buildFiltersUtilities {} {
    global flt_params
    set ma ""
    lappend ma "<E<SnewFilter"
    lappend ma "<S<InewMultiFilter"
    lappend ma "<E<SeditAFilter"
    lappend ma "<S<IshowMultiFilter"
    lappend ma "<S<BeditMultiFilter"
    lappend ma "<E<SdeleteAFilter"
    lappend ma "<S<IclearMultiFilter"
    lappend ma "$flt_params(prfx)checkSyntax"
    lappend ma "(-"
    lappend ma "updateFiltersList" 
    lappend ma "showFiltersBindings"
    lappend ma "displaySyntax"
    lappend ma "filtersTutorial"
    
    return [list build $ma flt::UtilsProc {}]
}


# # # Menu items procs # # #

proc flt::MenuProc {menu item} {
    global flt_params 
    switch -- $item {
	"applyToSelection" {
	    set flt_params(appliedFilter) $flt_params(currFiltername)
	    if {![flt::isFilterSelected]} {return}
	    set flt_params(ext) ".flt"
	    flt::applyToSelectionProc
	}
	"applyToFile" {
	    set flt_params(appliedFilter) $flt_params(currFiltername)
	    if {![flt::isFilterSelected]} {return}
	    set flt_params(ext) ".flt"
	    flt::applyToFileProc
	    set flt_params(firstapplic) 0
	}
	"applyToFolder..." {
	    set flt_params(appliedFilter) $flt_params(currFiltername)
	    if {![flt::isFilterSelected]} {return}
	    set flt_params(ext) ".flt"
	    if {![flt::getFolder]} {return}
	    flt::applyToFolderProc
	    set flt_params(firstapplic) 0
	}
	"applyMultiToFile" {
	    flt::applyMultiToFile
	    set flt_params(firstapplic) 0
	}
	"applyMultiToFolder..." {
	    flt::applyMultiToFolderProc
	    set flt_params(firstapplic) 0
	}
	"applyTempToFile" {
	    flt::applyTempToFileProc
	    set flt_params(firstapplic) 0
	}
	"applyTempToFolder..." {
	    flt::applyTempToFolderProc
	    set flt_params(firstapplic) 0
	}
	"applyTempToSelection" {flt::applyTempToSelectionProc}
	"applyMultiToSelection" {flt::applyMultiToSelection}
	"pickAFilter" {flt::pickAFilterProc}
	"buildAMultiFilter" {flt::newMultiFilter}
	"temporaryFilter" {flt::temporaryFilterProc}
	"currentFilter"  {flt::currFiltInfo}
	"noFilterSelected"  {flt::currFiltInfo}
    }
}

proc flt::UtilsProc {menu item} {
    switch -- $item {
        "filtersTutorial" {help::openExample "Filters Example"}
	default {eval flt::$item}
    }
}


	
# # # Building the menu # # #

menu::buildSome filtersMenu 


# # # Filters manipulation procs # # #

proc flt::pickAFilterProc {} {
    global flt_params
    flt::updateFiltersList
    if {[llength $flt_params(filtnameindir)] == 0} {
	alertnote "No filter in folder [file join Tcl Menus "Filters Menu" Filters]"
	return
    } else {
	set filt [listpick -p "Select a filter"  $flt_params(filtnameindir)]
	set flt_params(currFiltername) $filt
	menu::buildSome filtersMenu
    }
}

proc flt::deleteAFilter {} {
    global flt_params
    flt::updateFiltersList
    if {[llength $flt_params(filtnameindir)] == 0} {
	alertnote "No filter in folder [file join Tcl Menus "Filters Menu" Filters]"
	return
    } else {
	set filt [listpick -L $flt_params(currFiltername) -p "Filter to delete :"  $flt_params(filtnameindir)]
	if {[file exists [file join $flt_params(dirpath) $filt.flt]]} {
	    switch [buttonAlert "OK to delete filter \"$filt\" ?" "yes" "cancel" ] {
		"yes" {
		    file delete [file join $flt_params(dirpath) $filt.flt]
		    message "Filter $filt deleted."
		    flt::updateFiltersList
		    if {$flt_params(currFiltername) == $filt} {
			set flt_params(currFiltername) ""
			menu::buildSome filtersMenu
			}
		}
		"cancel" {return}
	    }
	} else {
	    alertnote "I can't find [file join $flt_params(dirpath) $filt.flt]"
	    return}
    }
}

proc flt::editAFilter {} {
    global flt_params
    flt::updateFiltersList
    if {[llength $flt_params(filtnameindir)] == 0} {
	alertnote "No filter in folder  [file join Tcl Menus "Filters Menu" Filters]"
	return
    } else {
	set filt [listpick -L $flt_params(currFiltername) -p "Filter to edit :"  $flt_params(filtnameindir)]
	if {[file exists [file join $flt_params(dirpath) $filt.flt]]} {
	    edit [file join $flt_params(dirpath) $filt.flt]
	    set flt_params(currFiltername) $filt
	    menu::buildSome filtersMenu
	    newMode Fltr
	} else {
	    alertnote "I can't find [file join $flt_params(dirpath) $filt.flt]"
	    return
	}
    }
}

proc flt::newFilter {} {
    global flt_params filtersMenumodeVars  
    catch {prompt "Name of the new filter (without extension)." NewFilter} fltname
    if {$fltname == "cancel"} {return} 
    set fileId [alphaOpen [file join $flt_params(dirpath) $fltname.flt] w+] 
    close $fileId
    if {[file exists [file join $flt_params(dirpath) $fltname.flt]]} {
	edit [file join $flt_params(dirpath) $fltname.flt]
    } else {
	alertnote "I can't find [file join $flt_params(dirpath) $fltname.flt]"
    }
    set date [ISOTime::ISODateAndTimeRelaxed]
    set t "!! Filter : $fltname.flt\n"
    append t "!! Created : $date\n"	
    append t "!! Description : \n!! \n!! \n!! \n\n"
    insertText $t
    if {$filtersMenumodeVars(showFilterSyntax)} {
	insertText $flt_params(filtersyntax)
    }
    lunion flt_params(filtersindir) [file tail $fltname]
    set flt_params(currFiltername) [file rootname [file tail $fltname]]
    menu::buildSome filtersMenu
    newMode Fltr
}

proc flt::updateFiltersList {} {
    global flt_params
    set flt_params(filtersindir) [glob -nocomplain -dir $flt_params(dirpath) *.flt]
    set flt_params(filtnameindir) ""
    foreach filt $flt_params(filtersindir) {lappend flt_params(filtnameindir) "[file rootname [file tail $filt]]"}
}

proc flt::temporaryFilterProc {} {
    global flt_params filtersMenumodeVars  
    global tileLeft tileTop tileWidth errorHeight
    if {[flt::checkTempOpen]} {
	bringToFront $flt_params(tempfltname)
    } else {
	if {[file exists [file join $flt_params(dirpath) $flt_params(tempfltname)]]} {
	    file delete [file join $flt_params(dirpath) $flt_params(tempfltname)]
	}
	set fileId [alphaOpen [file join $flt_params(dirpath) $flt_params(tempfltname)] w+] 
	close $fileId
	edit  -g $tileLeft $tileTop $tileWidth [expr {$errorHeight+80}] \
	  [file join $flt_params(dirpath) $flt_params(tempfltname)]
	newMode Fltr
	set t "!! In this window, you can define filtering instructions for a temporary use.\n"
	append t "!! Then bring to the foreground the window you want to apply it to\n"
	append t "!! and use one of the \"Apply Temp\" menu items.\n\n"
	insertText $t
	if {$filtersMenumodeVars(showFilterSyntax)} {
	    insertText $flt_params(filtersyntax)
	}
    }
}

proc flt::currFiltInfo {} {
    global flt_params
    if {$flt_params(currFiltername) == ""} {
	set mess "No filter currently selected.\r"
    } else {
	set mess "Current Filter is: $flt_params(currFiltername).\r"
    }
    if {![llength $flt_params(multiList)]} {
	append mess "MultiFilter empty."
    } else {
	append mess "MultiFilter contains : $flt_params(multiList)"
    }
    alertnote "$mess"
}

proc flt::togglesupsea {} {
    global filtersMenumodeVars flt_params alpha::platform
    if {${alpha::platform} != "alpha"} { return }
    message "Toggling superSearch [expr {$flt_params(supseatog) ? {on} : {off}}]"
    if {$filtersMenumodeVars(ToggleSupSearch) && [package::active supersearch]} {
	global supersearchOn
	if {$flt_params(supseatog)} {
	    set supersearchOn 1
	    supersearch::onoff
	    set flt_params(supseatog) 0
	} else {
	    set supersearchOn 0
	    supersearch::onoff
	    set flt_params(supseatog) 1
	}    
    }   
}


# # # Syntax procs # # #

proc flt::displaySyntax {} {
    global flt_params
    global tileLeft tileTop tileWidth errorHeight
    catch {lsearch -exact [winNames] $flt_params(syntfltname)} indx
    if {$indx > -1} {
	bringToFront $flt_params(syntfltname)
    } else {
	new -g $tileLeft $tileTop $tileWidth [expr int($errorHeight * 1.3)] \
	  -n "$flt_params(syntfltname)" -info  $flt_params(filtersyntax)
	flt::colorUsage
    }
}

proc flt::colorUsage {} {
    global flt_params
    catch {search -s -r 0 $flt_params(usage) [minPos]} res
    text::color [lindex $res 0] [lindex $res 1] 1
    set start [minPos]
    while {![catch {search -f 1 -s -r 1 "!!( |\t)+(0|1|i|m) " $start} res]} {
	text::color [pos::math [lindex $res 1]-2] [lindex $res 1] 1
	set start [lindex $res 1]
    }
    refresh
}

proc flt::checkSyntax {} {
    global flt_params filtersMenumodeVars
    global tileLeft tileTop tileWidth tileHeight    
    # First verify that the front window is a filter
    set winName  [win::CurrentTail]
    if { [file extension $winName] != ".flt" && [file rootname $winName] != $flt_params(tempfltname)} {
	alertnote "Current window is not a filter. It must have an \".flt\" extension."
	return
    }
    # Check that the filter's window is not dirty
    if {[winDirty]} {
	case [askyesno -c "Dirty filter \"[file rootname $winName]\". Do you want to save it before ?"] in {
	    "yes" {save}
	    "no" {}
	    "cancel" {return}
	}
    }
    
    # Verify that a "* Syntax Checking Results *" window is not already open
    # from a previous syntax checking. If any, remove it.
    catch {lsearch -exact [winNames] $flt_params(errfltname)} indx
    if {[expr {$indx > -1}]} {
	bringToFront $flt_params(errfltname)
	killWindow
	if {[file exists [file join $flt_params(dirpath) $flt_params(errfltname)]]} {
	    file delete [file join $flt_params(dirpath) $flt_params(errfltname)]
	}
    }
    # # Open the Error window. 
    set flt_params(errfileId) [alphaOpen [file join $flt_params(dirpath) $flt_params(errfltname)] w+] 
    # Open and read the filter's contents
    set filtId [alphaOpen [file join $flt_params(dirpath) $winName]] 
    catch {read $filtId} flt_params(textflt)
    close $filtId
    set flt_params(textflt) [split $flt_params(textflt) "\n"]
    if {![llength $flt_params(textflt)]} {return}
    # Parse each line
    set flt_params(lignenum) 0
    set flt_params(errnum) 0
    foreach line $flt_params(textflt) {
	set flt_params(line) $line
	set flt_params(lignenum) [expr {$flt_params(lignenum) + 1}]
	if { $flt_params(line) != "" && ![regexp {^!!} $flt_params(line) ]} {
	    # 	Splitting the flt_params(line)
	    flt::getSearchReplStrings
	    # 	No more than three args on a line :
	    if {[expr {[llength $flt_params(line)] > 3}]} {
		flt::errRecord "Too many args ([llength $flt_params(line)]) at line $flt_params(lignenum).\n\n"
	    }
	    if [flt::stopChecking] break
	    # First arg (searchString) must not be empty :
	        if {$flt_params(searchstr) == ""} {
		    flt::errRecord "First argument is empty at line $flt_params(lignenum).\n\n"
		}
	    if [flt::stopChecking] break
	    # 	Searching type $flt_params(grep) must be 0 or 1 (empty has alrady been made 0)
	    if {$flt_params(grep) != 0 && $flt_params(grep) != 1} {
		flt::errRecord "Wrong third arg at line $flt_params(lignenum) : \
		  $flt_params(grep).\nit should be 0, 1 or empty.\n\n"
	    } 
	    if [flt::stopChecking] break
	    if {$flt_params(grep) == 1} {
		# Check the patterns validity (borrowed from search.tcl)
		if {[catch {regexp -- $flt_params(searchstr) {} dmy} dmy]} {
		    flt::errRecord "Syntax error in pattern \"$flt_params(searchstr)\" \
		      for searchString at line $flt_params(lignenum) -\n$dmy\n\n"
		}
		if [flt::stopChecking] break
		
		if {[catch {regexp -- $flt_params(replstr) {} dmy} dmy]} {
		    flt::errRecord "Syntax error in pattern \"$flt_params(replstr)\" for searchString \
		      at line $flt_params(lignenum) -\n$dmy\n\n"
		} 
		if [flt::stopChecking] break
		
	    }	
	}
    }
    if {[expr {$flt_params(errnum) > 0}]} {
	set errmess "\n\n*** That makes $flt_params(errnum) errors. ***\n"
	if {[expr {$flt_params(errnum) > $filtersMenumodeVars(maxNumbErr) - 1}]} {
	    append errmess "Maximum number of errors is $filtersMenumodeVars(maxNumbErr) (see the Filters Menu prefs).\n"
	    append errmess "Syntax checking aborted.\n\nUsage :\n$flt_params(filtersyntax)"
	}
	puts $flt_params(errfileId) $errmess
    } else {
	puts $flt_params(errfileId) "*** Syntax OK ***"
    }
    close $flt_params(errfileId) 
    set top [expr {$tileTop + $tileHeight - 160}]
    edit -r -g [expr {$tileLeft + 10}] $top [expr {$tileWidth - 20}] 160  \
      [file join $flt_params(dirpath) $flt_params(errfltname)]
    flt::colorUsage
}

proc flt::errRecord {arg} {
    global flt_params
    puts $flt_params(errfileId) $arg
    set flt_params(errnum) [expr {$flt_params(errnum) + 1}]
}

proc flt::stopChecking {} {
    global flt_params filtersMenumodeVars  
    return [expr {$flt_params(errnum) > $filtersMenumodeVars(maxNumbErr) - 1}]
}

# # # MultiFilters manipulation procs # # #

proc flt::newMultiFilter {} {
    global flt_params
    flt::updateFiltersList
    flt::buildMultiFilter
}

proc flt::buildMultiFilter {} {
    global flt_params
    set args ""
    lappend args [list -t "* Multi filter building window *" 97 8 400 28 \
      -b OK 315 180 380 200 \
      -b Cancel 225 180 295 200 \
      -b Rebuild 135 180 205 200 \
      -b Add 33 51 83 71 \
      -t "Multi filter" 30 82 150 102 \
      -e "$flt_params(multiList)" 30 105 380 165
    ]
    set y 35
    eval lappend args [dialog::text "Filters " 105 y] \
      [list [dialog::menu 105 y $flt_params(filtnameindir) 0]]
    set flt_params(values) [eval dialog -w 390 -h 210 [join $args]]
    return [flt::getMultValues]
}

proc flt::getMultValues {} {
    global flt_params
    if {[lindex $flt_params(values) 1]} {return 0}
    if {[lindex $flt_params(values) 3]} {
	lappend flt_params(multiList) [lindex $flt_params(values) 5]
	flt::buildMultiFilter
	return 1
    }
    if {[lindex $flt_params(values) 2]} {
	flt::clearMultiFilter
	flt::buildMultiFilter
	return 1
	}
	set flt_params(multiList) [lindex $flt_params(values) 4]
    return 1	    
}

proc flt::showMultiFilter {} {
    global flt_params
    if {![llength $flt_params(multiList)]} {
	alertnote "Multifilter is empty."
	return
    }
    catch {set filt [listpick -p "List of filters in the current Multifilter. Confirm ?"  $flt_params(multiList)]} rep
    if {$rep == ""} {flt::buildMultiFilter}
}

proc flt::editMultiFilter {} {
    global flt_params
    if {![llength $flt_params(multiList)]} {
	alertnote "Multifilter is empty."
	return
    }
    foreach filt $flt_params(multiList) {
	edit [file join $flt_params(dirpath) $filt.flt]
    }
}

proc flt::clearMultiFilter {} {
    global flt_params
    set flt_params(multiList) ""
}


# # # Applying filters procs # # #

proc flt::applyToSelectionProc {} {
    global flt_params filtersMenumodeVars 
    set flt_params(debsel) [getPos]
    set flt_params(finsel) [selEnd]
    if {[pos::compare $flt_params(debsel) == $flt_params(finsel)]} {
	alertnote "No region selected."
	return
    }
    if {[expr {$filtersMenumodeVars(maxBeforeScrap) > [pos::diff $flt_params(finsel) $flt_params(debsel)]}]} {
	flt::filterTheSelection
	goto $flt_params(finsel)
	} else {
    select $flt_params(debsel) $flt_params(finsel)
    set flt_params(thesel) [getSelect]
    flt::filterOnScrap
    set flt_params(finsel) [pos::math $flt_params(debsel) + [string length $flt_params(thesel)]]
    goto $flt_params(finsel)
    }
    message "Filtering done."
}

proc flt::filterOnScrap {} {
    global flt_params   
    if {![flt::checkFiltDirty]} {return}
    if {[flt::filterToItself]} {return}
    if {![flt::getFilterData]} {return}
    deleteSelection
    flt::getScrapwindow
    flt::scrapFiltering
    flt::backFromScrapwindow
}

proc flt::getScrapwindow {} {
    global flt_params
    set flt_params(thewindow) [win::CurrentTail]
    catch {lsearch -exact [winNames] "$flt_params(scrapname)"} indx
    if {![expr {$indx > -1}]} {
	new -n $flt_params(scrapname) -g 4 50 20 20
    } else {
	bringToFront $flt_params(scrapname)
    }
    moveWin $flt_params(scrapname) 10000 10000
    insertText $flt_params(thesel)
}

proc flt::scrapFiltering {} {
    global flt_params 
    flt::togglesupsea
    flt::filterCore
    flt::togglesupsea
}

proc flt::backFromScrapwindow {} {
    global flt_params
    select [minPos] [maxPos]
    set flt_params(thesel) [getSelect]
    setWinInfo -w $flt_params(scrapname) dirty 0
    killWindow
    bringToFront $flt_params(thewindow)
    insertText $flt_params(thesel)    
}

# There is an unexplainable bug with the following proc when superSearch is active.
# Applying a filter to a short selection (less than MaxBeforeScrap) fails. 
# So, as a workaround, we deactivate/reactivate superSearch.
# I believe I've fixed that bug (Vince).
proc flt::filterTheSelection {} {
    global flt_params
    if {![flt::checkFiltDirty]} {return}
    if {[flt::filterToItself]} {return}
    if {![flt::getFilterData]} {return}
    foreach line $flt_params(textflt) {
	set flt_params(line) $line
	if { $flt_params(line) != "" && ![regexp {^!!} $flt_params(line)]} {
	    flt::getSearchReplStrings
	    flt::substitute $flt_params(searchstr) $flt_params(replstr) $flt_params(grep) \
	      $flt_params(ign) $flt_params(matchw)
	}
    }
}

proc flt::applyToFileProc {} {
    global flt_params filtersMenumodeVars 
    if {![flt::checkFiltDirty]} {return}
    if {[flt::filterToItself]} {return}
    if {$filtersMenumodeVars(warnUndoable) && !$flt_params(firstapplic)} {
	switch [buttonAlert "This is not undoable. Still want to apply the filter ?" "yes" "cancel" ] {
	    "yes" {}
	    "cancel" {return}
	}
    }
    set flt_params(firstapplic) [expr {$flt_params(firstapplic) + 1}]
    message "Applying \"$flt_params(appliedFilter)\" to [file tail [win::Current]]..."
    if {![flt::getFilterData]} {return}
    flt::togglesupsea
    flt::filterCore
    flt::togglesupsea
    message "Filtering done."
}

proc flt::filterToItself {} {
    global flt_params
    if {[file rootname [file tail [win::Current]]] == $flt_params(appliedFilter)} {
	switch [buttonAlert "Do you really want to apply $flt_params(appliedFilter) to itself ?" "no" "yes" ] {
	    "yes" {return 0}
	    "no" {return 1}
	}
    } else {
	return 0
    }
}    

proc flt::getFilterData {} {
    global flt_params 
    set fileId [alphaOpen [file join $flt_params(dirpath) $flt_params(appliedFilter)$flt_params(ext)]] 
    catch {read $fileId} textflt
    close $fileId
    set flt_params(textflt) [split $textflt "\n"]
    return [string length $textflt]
}

proc flt::getSearchReplStrings {} {
    global flt_params
    set prov [split $flt_params(line) "\t"]
    #     if there is only one arg on the line, consider that the replacement string
    #     is empty and the type $flt_params(grep) is 0 (no grep)
    if {[llength $prov] == 1} {
	set flt_params(searchstr) [lindex $prov 0]
	set flt_params(replstr) ""
	set flt_params(grep) 0
	set flt_params(ign) 0
	set flt_params(matchw) 0
	set flt_params(line) $prov
    } elseif {[llength $prov] == 3} {
	# We handle at first the case when there are exactly three tab-separated args
	# (in case the replacement string is empty) since we will replace below multiple tabs 
	# by a single one : if not, the second arg (when empty) would be stripped.
	set flt_params(searchstr) [lindex $prov 0]
	set flt_params(replstr) [lindex $prov 1]
	set thirdarg [lindex $prov 2]
	if {[regsub "1" $thirdarg {&} res]} {
	    set flt_params(grep) 1 
	} else {
	    set flt_params(grep) 0
	}
	set flt_params(ign) [regexp "i" $thirdarg ]
	set flt_params(matchw) [regexp "m" $thirdarg ]
	set flt_params(line) $prov
    } else {
	# There may be several tabs between the args (to allow a better visual disposition
	# in the filters. We first replace them by a single one before splitting.
	regsub -all "\t+" $flt_params(line) "\t" flt_params(line)
	set flt_params(line) [split $flt_params(line) "\t"]
	set flt_params(searchstr) [lindex $flt_params(line) 0]
	set flt_params(replstr) [lindex $flt_params(line) 1]
	if {[expr {[llength $flt_params(line)] > 2}]} {
	    set thirdarg [lindex $flt_params(line) 2]
	} else {
	    set thirdarg ""
	}
	if {[regsub "1" $thirdarg {&} res]} {
	    set flt_params(grep) 1 
	} else {
	    set flt_params(grep) 0
	}
	set flt_params(ign) [regexp "i" $thirdarg ]
	set flt_params(matchw) [regexp "m" $thirdarg ]
    }
}    

proc flt::filterCore {} {
    global flt_params
    foreach line $flt_params(textflt) {
	set flt_params(line) $line
	if { $flt_params(line) != "" && ![regexp {^!!} $flt_params(line) ]} {
	    flt::getSearchReplStrings
	    flt::substituteAll $flt_params(searchstr) $flt_params(replstr) $flt_params(grep) \
	      $flt_params(ign) $flt_params(matchw)
	}
    }
}

proc flt::substitute {searchstr {replstr ""} {fltgrep 0} {fltign 0} {fltmatchw 0}} {
    global flt_params
    if {$flt_params(searchstr) == ""} {return}
    set start $flt_params(debsel)
    set end $flt_params(finsel)
    while {![catch {performSearch -f 1 -r $fltgrep -m $fltmatchw \
      -i $fltign -l $end $searchstr $start} res]} {
	replaceString $replstr
	set end [pos::math $end - [pos::diff [selEnd] [getPos]]]
	set start [getPos]
	replace
	set end [pos::math $end + [pos::diff [getPos] $start]]
	set start [getPos]
    }
    set flt_params(finsel) $end
}

proc flt::substituteAll {searchstr {replstr ""} {fltgrep 0} {fltign 0} {fltmatchw 0}} {
    global flt_params
    if {$searchstr == ""} {return}
    catch {
	replaceString $replstr
	performSearch -f 1 -r $fltgrep -m $fltmatchw -i $fltign -- $searchstr [minPos]
	replaceAll
    }
}

proc flt::isFilterSelected {} {
    global flt_params
    if {$flt_params(appliedFilter) == "" } {
	alertnote "No filter currently selected. Use the \"Pick a filter\" menu item."
	return 0
    } else {
	return 1
    }
}

proc flt::checkFiltDirty {} {
    global flt_params
    catch {lsearch -exact [winNames] "$flt_params(appliedFilter).flt"} indx
    if {![expr {$indx > -1}]} {
	return 1
    } else {
	getWinInfo -w "$flt_params(appliedFilter).flt" arr
	set mywindow [win::CurrentTail]
	if {$arr(dirty)} {
	    switch [buttonAlert "Dirty Filter \"$flt_params(appliedFilter)\". \
	      Do you want to save it ?" "yes" "no" "cancel"] {
		"yes" {
		    bringToFront "$flt_params(appliedFilter).flt"
		    save
		    bringToFront $mywindow
		    return 1
		}
		"no" {return 1}
		"cancel"  {return 0}
	    }
	} else {
	    return 1
	}
    }
}

# # # Applying temporary filter procs # # #

proc flt::applyTempToSelectionProc {} {
    global flt_params
    if {![flt::checkTempOpen]} {
	alertnote "No temporary Filter."
	return
    }
    flt::checkTempDirty
    set flt_params(appliedFilter) $flt_params(tempfltname)
    set flt_params(ext) ""
    flt::applyToSelectionProc
}

proc flt::applyTempToFileProc {} {
    global flt_params
    if {![flt::checkTempOpen]} {
	alertnote "No temporary Filter."
	return
    }
    flt::checkTempDirty 
    set flt_params(appliedFilter) $flt_params(tempfltname)
    set flt_params(ext) ""
    flt::applyToFileProc
}

proc flt::checkTempDirty {} {
    global flt_params
    getWinInfo -w $flt_params(tempfltname) arr
    set mywindow [win::CurrentTail]
    if {$arr(dirty)} {
	switch [buttonAlert "Dirty Temporary Filter. Do you want to save it ?" "yes" "no"  ] {
	    "yes" {
		bringToFront $flt_params(tempfltname)
		save
		bringToFront $mywindow
	    }
	    "no" {}
	}
    }
}

proc flt::checkTempOpen {} {
    global flt_params
    catch {lsearch -exact [winNames] $flt_params(tempfltname)} indx
    if {$indx > -1} {
	return 1
    } else {
	return 0
    }
}


# # # Applying multi filters procs # # #

proc flt::applyMultiToSelection {} {
    global flt_params filtersMenumodeVars 
    set flt_params(debsel) [getPos]
    set flt_params(finsel) [selEnd]
    if {$flt_params(debsel) == $flt_params(finsel)} {
	alertnote "No region selected."
	return
    }
    if {![llength $flt_params(multiList)]} {
	alertnote "Multifilter is empty."
	return
    }
    set flt_params(ext) ".flt"
    if {[expr {$filtersMenumodeVars(maxBeforeScrap) > [pos::diff $flt_params(finsel) $flt_params(debsel)]}]} {
	flt::applyMultiOnSelection
	} else {
    select $flt_params(debsel) $flt_params(finsel)
    set flt_params(thesel) [getSelect]
    deleteSelection
    flt::getScrapwindow
    foreach filt $flt_params(multiList) {
	set flt_params(appliedFilter) $filt
	if {![flt::getFilterData]} {continue}
	flt::scrapFiltering
    }
    flt::backFromScrapwindow
    set flt_params(finsel) [pos::math $flt_params(debsel) + [string length $flt_params(thesel)]]
    goto $flt_params(finsel)
    }
}

proc flt::applyMultiOnSelection {} {
    global flt_params
    foreach filt $flt_params(multiList) {
	set flt_params(appliedFilter) $filt
	flt::filterTheSelection
    }
    goto $flt_params(finsel)
}

proc flt::applyMultiToFile {} {
    global flt_params
    if {![llength $flt_params(multiList)]} {
	alertnote "Multifilter is empty."
	return
    }
    set flt_params(ext) ".flt"
    foreach filt $flt_params(multiList) {
	set flt_params(appliedFilter) $filt
	flt::applyToFileProc
    }
}


# # # Applying filters to folders procs # # #

proc flt::applyToFolderProc {} {
    global flt_params flt_matches flt_case filtersMenumodeVars win::NumDirty

    if {![flt::checkFiltDirty]} {return}
    if {$filtersMenumodeVars(warnUndoable) && !$flt_params(firstapplic)} {
	switch [buttonAlert "This is not undoable. Still want to apply the filter ?" "yes" "cancel" ] {
	    "yes" {}
	    "cancel" {return}
	}
    }
    set flt_params(firstapplic) [expr {$flt_params(firstapplic) + 1}]
    if {![flt::getFilterData]} {return}
    
    # Check that there are no dirty windows since scanning is done on the disk (from filesets.tcl)
    if {${win::NumDirty}} {
	if {[buttonAlert "Save all windows?" "Yes" "Cancel"] != "Yes"} return
	saveAll
    }
    
    # scan the files to mark those containing a match :
    set flt_params(cid) [scancontext create]
    flt::buildScanMatches
    flt::scanFiles
    scancontext delete $flt_params(cid)
    
    # Now the filter will be applied only to those files for which there has been a match :
    flt::filterTheMatches
    if {[info exists flt_matches]} {unset flt_matches}
    message "Filtering done."
}

proc flt::buildScanMatches {} {
    global flt_params flt_matches flt_case 
    # one scanmatch command for each line in the filter :
    foreach line $flt_params(textflt) {
	set flt_params(line) $line
	if { $flt_params(line) != "" && ![regexp {^!!} $flt_params(line) ]} {
	    flt::getSearchReplStrings
	    if {$flt_params(grep) == 0} {
		set flt_params(searchstr) [quote::Regfind $flt_params(searchstr)]
	    }
	    scanmatch $flt_case($flt_params(ign)) $flt_params(cid) $flt_params(searchstr) {set flt_matches($f) 1}
	}
    }
}

proc flt::scanFiles {} {
    global flt_params flt_matches flt_case 
    # Scan all the files looking for matches :
    foreach f $flt_params(TextFilesindir) {
	if {![catch {set fid [alphaOpen $f]}]} {
	    message "Looking at '[file tail $f]'"
	    scanfile $flt_params(cid) $fid
	    close $fid
	}
    } 
}

proc flt::filterTheMatches {} {
    global flt_params flt_matches
    flt::togglesupsea
    foreach f [lsort [array names flt_matches]] {
	if {[file exists "$f"]} {
	    set ftail [file tail $f]
	    catch {lsearch -exact [winNames] $ftail} indx
	    if {[expr {$indx > -1}]} {
		bringToFront $ftail
	    } else {
		edit -w "$f"
	    }
	    if {![flt::filterToItself]} {		
		message "Applying \"$flt_params(appliedFilter)\" to $ftail..."
		flt::filterCore
	    }
	} else {
	    alertnote "I can't edit $f"
	}
    }
    flt::togglesupsea
}

proc flt::applyMultiToFolderProc {} {
    global flt_params flt_matches flt_case filtersMenumodeVars win::NumDirty
    watchCursor
    if {![llength $flt_params(multiList)]} {
	alertnote "Multifilter is empty."
	return
    }
    if {![flt::getFolder]} {return}
    if {$filtersMenumodeVars(warnUndoable) && !$flt_params(firstapplic)} {
	switch [buttonAlert "This is not undoable. Still want to apply the filter ?" "yes" "cancel" ] {
	    "yes" {}
	    "cancel" {return}
	}
    }
    set flt_params(ext) ".flt"
    set flt_params(firstapplic) [expr {$flt_params(firstapplic) + 1}]
    
    # Check that there are no dirty windows before scanning
    if {${win::NumDirty}} {
	if {[buttonAlert "Save all windows?" "Yes" "Cancel"] != "Yes"} return
	saveAll
    }
    
    # scan the files to mark those containing a match :
    set flt_params(cid) [scancontext create]
    flt::buildMultiScanMatches
    flt::scanFiles
    scancontext delete $flt_params(cid)
    
    # Now each filter will be applied only to those files for which there has been a match :
    foreach filt $flt_params(multiList) {
	set flt_params(appliedFilter) $filt
	if {![flt::getFilterData]} {continue}
	flt::filterTheMatches
    }
    if {[info exists flt_matches]} {unset flt_matches}
}

proc flt::buildMultiScanMatches {} {
    global flt_params flt_matches flt_case
    # here all the filters instructions are gathered in one scan context :
    foreach filt $flt_params(multiList) {
	set flt_params(appliedFilter) $filt
	if {[flt::getFilterData]} {
	    if {![flt::checkFiltDirty]} {return}
	    # one scanmatch command for each line in all the filters :
	    foreach line $flt_params(textflt) {
		set flt_params(line) $line
		if { $flt_params(line) != "" && ![regexp {^!!} $flt_params(line) ]} {
		    flt::getSearchReplStrings
		    if {$flt_params(grep) == 0} {
			set flt_params(searchstr) [quote::Regfind $flt_params(searchstr)]
		    }
		    scanmatch $flt_case($flt_params(ign)) $flt_params(cid) $flt_params(searchstr) {set flt_matches($f) 1}
		}
	    }
	}
    }
}

proc flt::applyTempToFolderProc {} {
    global flt_params
    if {![flt::checkTempOpen]} {
	alertnote "No temporary Filter."
	return
    }
    if {![flt::getFolder]} {return}
    flt::checkTempDirty    
    set flt_params(appliedFilter) $flt_params(tempfltname)
    set flt_params(ext) ""
    flt::applyToFolderProc
}

proc flt::getFolder {} {
    global flt_params
    catch {get_directory -p "Select a folder."} fltfolderpath
    if {$fltfolderpath == ""} {return 0}
    set flt_params(TextFilesindir) [glob -types TEXT -nocomplain -dir $fltfolderpath *]
    if {![llength $flt_params(TextFilesindir)]} {
	alertnote "No Text files in this folder."
	return 0
    } else {
	return 1
    }
}

# # # Exporting procs # # #
# Here we define two procs to export the filtering capacity in other scripts.
# Introduced in version 1.4.
# Just say "alpha::package require filtersMenu 1.4" at the beginning of your script
# and then use one of the following commands
#     flt::filterThisFile $filtername $filename
#     flt::filterThisSelection $filtername $selection
# If filename is not specified then it applies to the current window.

proc flt::filterThisFile {filtername {filename ""}} {
    global flt_params
    if {$filename == ""} {
	set filename [win::Current]
    } elseif {[file isfile $filename]} {
	edit -c -w $filename
    } elseif {[catch {bringToFront $filename}]} {
	status::errorMsg "Cannot find the filename '$filename'"
    }
    set flt_params(appliedFilter) $filtername
    set flt_params(ext) ".flt"
    if {![file exists [file join $flt_params(dirpath) $filtername$flt_params(ext)]]} {
	alertnote "Can't find filter $filtername."
        return 0
    } 
    if {![flt::getFilterData]} {return 0}
    flt::togglesupsea
    flt::filterCore
    flt::togglesupsea
    return 1
}

proc flt::filterThisSelection {filtername selection} {
    global flt_params
    set flt_params(appliedFilter) $filtername
    set flt_params(ext) ".flt"
    if {![file exists [file join $flt_params(dirpath) $filtername$flt_params(ext)]]} {
	return ""
    } 
    # Do the filtering in a scrap window
    set flt_params(thesel) $selection
    flt::getScrapwindow
    if {![flt::getFilterData]} {return ""}
    flt::togglesupsea
    flt::filterCore
    flt::togglesupsea
    # Retrieve the result
    select [minPos] [maxPos]
    set flt_params(thesel) [getSelect]
    setWinInfo -w $flt_params(scrapname) dirty 0
    killWindow
    return $flt_params(thesel)
}

# # # Miscellaneous # # #

proc flt::showFiltersBindings {} {
    global tileLeft tileTop tileWidth errorHeight
	    set mess "KEY BINDINGS AVAILABLE FOR THE FILTERS MENU\n\n"
	    append mess "Press 'ctrl-f', release, then hit one of the following letters :\n"
	    append mess "  'b'  to show the <b>indings\n"
	    append mess "  'c'  to <c>heck the filter's syntax\n"
	    append mess "  'd'  to apply filter to a fol<d>er (or <d>irectory)\n"
	    append mess "  'e'  to <e>dit a filter\n"
	    append mess "  'f'  to apply filter to the current <f>ile\n"
	    append mess "  'm'  to build a <m>ultifilter\n"
	    append mess "  'n'  to create a <n>ew filter\n"
	    append mess "  'p'  to <p>ick a filter\n"
	    append mess "  's'  to apply filter to a <s>election\n"
	    append mess "  't'  to call up the <t>emporary filter\n"
	    append mess "  'shift-d'  to apply multifilter to a fol<d>er (or <d>irectory)\n"
	    append mess "  'shift-e'  to <e>dit a multifilter\n"
	    append mess "  'shift-f'  to apply multifilter to the current <f>ile\n"
	    append mess "  'shift-s'  to apply multifilter to the <s>election\n"
	    append mess "\nPress 'ctrl-t', release, then hit one of the following letters :\n"
	    append mess "  'd'  to apply temporary filter to a fol<d>er\n"
	    append mess "  'f'  to apply temporary filter to the current <f>ile\n"
	    append mess "  's'  to apply temporary filter to a <s>election\n"
	    new -g $tileLeft $tileTop [expr int($tileWidth*.5)] \
	      [expr int($errorHeight *1.8)] \
	      -n "* Filters Bindings *" -info $mess
	    set start [minPos]
	    while {![catch {search -f 1 -s -r 1 {('|<)[a-z-]+('|>)} $start} res]} {
		text::color [lindex $res 0] [lindex $res 1] 1
		set start [lindex $res 1]
	    }
	    text::color [minPos] [nextLineStart [minPos]] 5
	    refresh
}

##########    Key bindings   ############
# The following instructions install easy to remember keybindings " la emacs".
#  For all of them you have to hit 'ctrl-f', release, then hit one of the following letters :
#  -  p  to <p>ick a filter
#  -  s  to filter the <s>election
#  -  f  to filter the current <f>ile
#  -  d  to filter a fol<d>er (or <d>irectory)
#  -  e  to <e>dit a filter
#  -  n  to create a <n>ew filter
#  -  t  to call up the <t>emporary filter
#  -  c  to <c>heck the syntax
#  -  m  to build a <m>ultifilter
#  -  h  to display the short syntax <h>elp
#  -  w  to show <w>hich is the active filter
#  -  b  to show the <b>indings

# Now if you add the shift key with letters s, f, d, e you get the equivalent with Multifilter
# instead of Filter. For instance 'ctrl-f shift-s' is equivalent to "Apply Multifilter to the Selection".

# There are three more key bindings to use the Temporary Filter. First hit 'ctrl-t', release, then hit 
# one of the letters s, f, d to apply the temporary filter to a <s>election, to the current <f>ile 
# or to a <d>irectory respectively.

Bind 'f' <z> prefixChar 
Bind 't' <z> prefixChar 
Bind 'c' <F> {flt::checkSyntax}
Bind 'c' <sF> {flt::clearMultiFilter}
Bind 'd' <F> {flt::MenuProc "filtersMenu" "applyToFolder..."}
Bind 'd' <sF> {flt::MenuProc "filtersMenu" "applyMultiToFolder..."}
Bind 'd' <T> {flt::MenuProc "filtersMenu" "applyTempToFolder..."}
Bind 'e' <F> {flt::editAFilter}
Bind 'e' <sF> {flt::editMultiFilter}
Bind 'f' <F> {flt::MenuProc "filtersMenu" "applyToFile"}
Bind 'f' <sF> {flt::MenuProc "filtersMenu" "applyMultiToFile"}
Bind 'f' <T> {flt::MenuProc "filtersMenu" "applyTempToFile"}
Bind 'h' <F> {flt::displaySyntax}
Bind 'm' <F> {flt::newMultiFilter}
Bind 'n' <F> {flt::newFilter}
Bind 'p' <F> {flt::pickAFilterProc}
Bind 's' <F> {flt::MenuProc "filtersMenu" "applyToSelection"}
Bind 's' <sF> {flt::MenuProc "filtersMenu" "applyMultiToSelection"}
Bind 's' <T> {flt::MenuProc "filtersMenu" "applyTempToSelection"}
Bind 't' <F> {flt::temporaryFilterProc}
Bind 'w' <F> {flt::currFiltInfo}
Bind 'b' <F> {flt::showFiltersBindings}

##########   Option-click on title bar   ############
# If you Option-Click on a the title bar of a filter, you get a list  of  all
# the filters stored in the  "Menus:Filters Menu:Filters:"  folder.  Selecting
# any item will open it in a window or bring its window to  front  if  it  is
# already open, and will make it the current filter as shown at the bottom of
# the Filters menu.

namespace eval Fltr {}

proc Fltr::OptionTitlebar {} {
    global flt_params 
    return $flt_params(filtnameindir)
}

proc Fltr::OptionTitlebarSelect {item} {
    global flt_params
    if {[file exists [file join [file dirname [win::Current]] $item]]} {
	Fltr::BringOrEdit [file join [file dirname [win::Current]] "$item.flt"] -c
    } else {
	Fltr::BringOrEdit [file join $flt_params(dirpath) "$item.flt"] -c
    }
    set flt_params(currFiltername) $item
    menu::buildSome filtersMenu
    newMode Fltr
}

proc Fltr::BringOrEdit {name {type ""}} {
    catch {lsearch -exact [winNames] [file tail $name]} indx
    if {[expr {$indx > -1}]} {
	bringToFront [file tail $name]
    } else {
	edit $type $name
    }
}

##########    The "M" menu   ############
proc Fltr::MarkFile {} {
    setNamedMark "go to top" [minPos] [minPos] [minPos]
    setNamedMark "go to bottom" [maxPos] [maxPos] [maxPos]
}

##########    The "{}" pop-up menu   ############
# The "{}" pop-up menu gives a count of the instruction lines and of
# the comments in a filter file.
proc Fltr::parseFuncs {} {
    set pos [minPos]
    set m 0
    while {[set res [search -s -f 1 -r 1 -n "^!!" $pos]] != ""} {
	incr m
	set pos [lindex $res 1]
    }
    set pos [minPos]
    set l 0
    while {[set res [search -s -f 1 -r 1 -n "^\[^\r\]+$" $pos]] != ""} {
	incr l
	set pos [lindex $res 1]
    }
    return [list "[expr $l-$m] instructions" "" "$m comments" ""]
}
